<?php
namespace app\controllers;

use Yii;
use yii\web\Controller;
use yii\filters\AccessControl;
use app\models\Evaluation;
use app\models\Application;
use app\models\Contest;
use app\models\Nomination;
use app\models\AgeCategory;
use yii\data\ActiveDataProvider;
use yii\web\NotFoundHttpException;
use yii\filters\VerbFilter;

class ExpertController extends Controller
{
    public function behaviors()
    {
        return [
            'access' => [
                'class' => AccessControl::class,
                'rules' => [
                    [
                        'allow' => true,
                        'roles' => ['@'],
                        'matchCallback' => function($rule, $action) {
                            return Yii::$app->user->identity->role === 'expert';
                        }
                    ]
                ]
            ],
            'verbs' => [
                'class' => VerbFilter::class,
                'actions' => [
                    'evaluate' => ['POST', 'GET'],
                ],
            ],
        ];
    }

    /**
     * Список назначенных эксперту заявок с фильтрами
     */
    public function actionIndex()
    {
        $userId = Yii::$app->user->id;

        $query = Application::find()
            ->alias('a')
            ->innerJoin('expert_assignment ea', 'ea.contest_id = a.contest_id')
            ->andWhere(['ea.expert_id' => $userId])
            ->with(['contest', 'nomination', 'ageCategory']);

        // Учет назначения на все номинации
        $query->andWhere("
        ea.nomination_id IS NULL
        OR ea.nomination_id = a.nomination_id
    ");

        // Учет назначения на все возрастные категории
        $query->andWhere("
        ea.age_category_id IS NULL
        OR ea.age_category_id = a.age_category_id
    ");

        // Фильтры
        $request = Yii::$app->request;

        if ($contest = $request->get('contest_id')) {
            $query->andWhere(['a.contest_id' => $contest]);
        }

        if ($nom = $request->get('nomination_id')) {
            $query->andWhere(['a.nomination_id' => $nom]);
        }

        if ($age = $request->get('age_category_id')) {
            $query->andWhere(['a.age_category_id' => $age]);
        }

        // Фильтр по статусу оценки
        if ($status = $request->get('eval_status')) {
            $query->innerJoin('evaluation e', 'e.application_id = a.id AND e.expert_id = :exp', [':exp'=>$userId]);
            $query->andWhere(['e.status' => $status]);
        }

        $dataProvider = new ActiveDataProvider([
            'query' => $query,
            'pagination' => ['pageSize' => 20],
            'sort' => ['defaultOrder' => ['created_at' => SORT_DESC]],
        ]);

        return $this->render('index', [
            'dataProvider' => $dataProvider,
            'contests' => Contest::find()->all(),
            'nominations' => Nomination::find()->all(),
            'ageCategories' => AgeCategory::find()->all(),
        ]);
    }

    /**
     * Просмотр заявки + оценка (форма). Если POST — сохраняем draft / completed.
     */
    public function actionEvaluate($id)
    {
        $userId = Yii::$app->user->id;

        $evaluation = Evaluation::findOne(['application_id' => $id, 'expert_id' => $userId]);

        if (!$evaluation) {
            throw new NotFoundHttpException('Оценка для этой заявки не найдена или вам не назначена.');
        }

        // Блокировка редактирования законченной оценки
        if ($evaluation->status === 'completed') {
            Yii::$app->session->setFlash('info', 'Эта оценка завершена и недоступна для изменения.');
            return $this->redirect(['/expert/index', 'id' => $id]);
        }

        $application = $evaluation->application;

        if (Yii::$app->request->isPost) {

            $post = Yii::$app->request->post('Evaluation', []);

            $evaluation->mastery = $post['mastery'] ?? $evaluation->mastery;
            $evaluation->artistry = $post['artistry'] ?? $evaluation->artistry;
            $evaluation->stage_culture = $post['stage_culture'] ?? $evaluation->stage_culture;
            $evaluation->notes = $post['notes'] ?? $evaluation->notes;

            if (Yii::$app->request->post('save_draft') !== null) {
                $evaluation->status = 'draft';
            } else {
                $evaluation->status = 'completed';
            }

            if ($evaluation->save()) {

                // проверяем, все ли эксперты завершили оценку
                if ($evaluation->status === 'completed') {

                    $allFinished = Evaluation::find()
                            ->where(['application_id' => $id])
                            ->andWhere(['<>', 'status', 'completed'])
                            ->count() == 0;

                    if ($allFinished) {
                        $application->status = 'evaluated';
                        $application->save(false);
                    }
                }

                Yii::$app->session->setFlash('success', 'Оценка сохранена.');
                return $this->refresh();
            }

            Yii::$app->session->setFlash('error', json_encode($evaluation->errors));
        }

        return $this->render('evaluate', [
            'application' => $application,
            'evaluation' => $evaluation,
        ]);
    }





}
