<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

class User extends ActiveRecord implements IdentityInterface
{

    public function block()
    {
        $this->status = 'blocked';
        return $this->save();
    }

    const STATUS_ACTIVE = 'active';
    const STATUS_BLOCKED = 'blocked';
    const STATUS_INACTIVE = 'inactive';
    const ROLE_PARTICIPANT = 'participant';
    const ROLE_EXPERT = 'expert';
    const ROLE_ADMIN = 'admin';

    public static function tableName()
    {
        return 'user';
    }

    public function rules()
    {
        return [
            [['name', 'surname', 'login', 'email', 'password_hash'], 'required'],
            [['patronymic'], 'safe'],
            [['login', 'email'], 'unique'],
            ['email', 'email'],
            ['role', 'in', 'range' => [
                self::ROLE_PARTICIPANT,
                self::ROLE_EXPERT,
                self::ROLE_ADMIN
            ]],
        ];
    }


    public function setPassword(string $password)
    {
        $this->password_hash = Yii::$app->security->generatePasswordHash($password);
    }

    public function validatePassword(string $password)
    {
        return Yii::$app->security->validatePassword(
            $password,
            $this->password_hash
        );
    }

    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]); // Проверка, что пользователь активен
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return static::findOne(['auth_key' => $token, 'status' => self::STATUS_ACTIVE]);
    }

    public static function findByUsername($username)
    {
        return static::findOne(['login' => $username, 'status' => self::STATUS_ACTIVE]);  // Фильтруем по статусу
    }

    public static function findByLoginOrEmail($login)
    {
        return static::find()
            ->where(['login' => $login])
            ->orWhere(['email' => $login])
            ->one();
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAuthKey()
    {
        return null;
    }


    public function validateAuthKey($authKey)
    {
        return true;
    }

    public function beforeLogin($event)
    {
        if ($this->status === 'blocked') {
            Yii::$app->user->logout(); // Выход, если пользователь заблокирован
            Yii::$app->session->setFlash('error', 'Ваш аккаунт заблокирован.');
            return false; // Отказ от входа
        }
    }
}
